/*
   Setup / repair script for level 9 of the Stairway to Columnstore Indexes;
    for use when you lost the database or made too many changes to follow the next levels.
   After running this script, your database should be in more or less the same state
    as it was after running all demo scripts in levels 1 through 8.

   In the "restore database" statement below, change the path to the location
    where you have stored the Contoso database backup downloaded from Microsoft;
    also change the locations for the data and log files to match your system.

   If you lost your download, you can download a fresh copy of the ContosoDW database
    backup file from https://www.microsoft.com/en-us/download/details.aspx?id=18279
*/

-- First drop the existing database if it exists
USE master;
GO

IF EXISTS (SELECT * FROM sys.databases WHERE name = 'ContosoRetailDW')
BEGIN;
    ALTER DATABASE ContosoRetailDW SET SINGLE_USER WITH ROLLBACK IMMEDIATE;
    DROP DATABASE ContosoRetailDW;
END;

-- Now restore. Don't forget to edit path names in this statement!!
RESTORE DATABASE ContosoRetailDW
    FROM DISK = N'C:\Program Files\Microsoft SQL Server\MSSQL12.SQL2014\MSSQL\Backup\ContosoRetailDW.bak'
    WITH MOVE N'ContosoRetailDW2.0' TO N'C:\Program Files\Microsoft SQL Server\MSSQL12.SQL2014\MSSQL\DATA\ContosoRetailDW.mdf',
         MOVE N'ContosoRetailDW2.0_log' TO N'C:\Program Files\Microsoft SQL Server\MSSQL12.SQL2014\MSSQL\DATA\ContosoRetailDW.ldf',
         STATS = 5;
GO

USE ContosoRetailDW;
GO

-- Clustered index is tied to primary key, so we have to drop that first
ALTER TABLE dbo.FactOnlineSales
DROP CONSTRAINT PK_FactOnlineSales_SalesKey;

-- Now create a clustered index on the desired column
CREATE CLUSTERED INDEX ix_FactOnlineSales_ProductKey
ON dbo.FactOnlineSales(ProductKey);

-- Recreate the PRIMARY KEY constraint, now using a nonclustered supporting index
ALTER TABLE dbo.FactOnlineSales
ADD CONSTRAINT PK_FactOnlineSales_SalesKey
    PRIMARY KEY (OnlineSalesKey);

-- Then recreate the nonclustered columnstore index without parallelism
CREATE NONCLUSTERED COLUMNSTORE INDEX NCI_FactOnlineSales
ON dbo.FactOnlineSales
   (OnlineSalesKey, DateKey, StoreKey, ProductKey, PromotionKey, CurrencyKey,
    CustomerKey, SalesOrderNumber, SalesOrderLineNumber, SalesQuantity, SalesAmount,
    ReturnQuantity, ReturnAmount, DiscountQuantity, DiscountAmount, TotalCost,
    UnitCost, UnitPrice, ETLLoadID, LoadDate, UpdateDate)
WITH (MAXDOP = 1);

-- Create a copy of the table that we can create a clustered columnstore index on
SELECT *
INTO   dbo.FactOnlineSales2
FROM   dbo.FactOnlineSales;

-- Before creating the clustered columnstore, first re-do all modifications.
-- These should run a bit faster if done before creating the index.

-- Add 500 rows
INSERT     dbo.FactOnlineSales2
          (DateKey, StoreKey, ProductKey, PromotionKey,
           CurrencyKey, CustomerKey, SalesOrderNumber,
           SalesOrderLineNumber, SalesQuantity, SalesAmount,
           ReturnQuantity, ReturnAmount, DiscountQuantity,
           DiscountAmount, TotalCost, UnitCost, UnitPrice,
           ETLLoadID, LoadDate, UpdateDate)
SELECT TOP(500)
           DateKey, StoreKey, ProductKey, PromotionKey,
           CurrencyKey, CustomerKey, SalesOrderNumber,
           SalesOrderLineNumber, SalesQuantity, SalesAmount,
           ReturnQuantity, ReturnAmount, DiscountQuantity,
           DiscountAmount, TotalCost, UnitCost, UnitPrice,
           ETLLoadID, LoadDate, UpdateDate
FROM       dbo.FactOnlineSales;

-- Add another 150,000 rows
INSERT     dbo.FactOnlineSales2
          (DateKey, StoreKey, ProductKey, PromotionKey,
           CurrencyKey, CustomerKey, SalesOrderNumber,
           SalesOrderLineNumber, SalesQuantity, SalesAmount,
           ReturnQuantity, ReturnAmount, DiscountQuantity,
           DiscountAmount, TotalCost, UnitCost, UnitPrice,
           ETLLoadID, LoadDate, UpdateDate)
SELECT TOP(150000)
           DateKey, StoreKey, ProductKey, PromotionKey,
           CurrencyKey, CustomerKey, SalesOrderNumber,
           SalesOrderLineNumber, SalesQuantity, SalesAmount,
           ReturnQuantity, ReturnAmount, DiscountQuantity,
           DiscountAmount, TotalCost, UnitCost, UnitPrice,
           ETLLoadID, LoadDate, UpdateDate
FROM       dbo.FactOnlineSales;

-- And yet another 15,000 rows
INSERT     dbo.FactOnlineSales2
          (DateKey, StoreKey, ProductKey, PromotionKey,
           CurrencyKey, CustomerKey, SalesOrderNumber,
           SalesOrderLineNumber, SalesQuantity, SalesAmount,
           ReturnQuantity, ReturnAmount, DiscountQuantity,
           DiscountAmount, TotalCost, UnitCost, UnitPrice,
           ETLLoadID, LoadDate, UpdateDate)
SELECT TOP(15000)
           DateKey, StoreKey, ProductKey, PromotionKey,
           CurrencyKey, CustomerKey, SalesOrderNumber,
           SalesOrderLineNumber, SalesQuantity, SalesAmount,
           ReturnQuantity, ReturnAmount, DiscountQuantity,
           DiscountAmount, TotalCost, UnitCost, UnitPrice,
           ETLLoadID, LoadDate, UpdateDate
FROM       dbo.FactOnlineSales;
GO

-- Add another 1,100,000 rows, in 11 batches of 100,000 each
INSERT     dbo.FactOnlineSales2
          (DateKey, StoreKey, ProductKey, PromotionKey,
           CurrencyKey, CustomerKey, SalesOrderNumber,
           SalesOrderLineNumber, SalesQuantity, SalesAmount,
           ReturnQuantity, ReturnAmount, DiscountQuantity,
           DiscountAmount, TotalCost, UnitCost, UnitPrice,
           ETLLoadID, LoadDate, UpdateDate)
SELECT TOP(100000)
           DateKey, StoreKey, ProductKey, PromotionKey,
           CurrencyKey, CustomerKey, SalesOrderNumber,
           SalesOrderLineNumber, SalesQuantity, SalesAmount,
           ReturnQuantity, ReturnAmount, DiscountQuantity,
           DiscountAmount, TotalCost, UnitCost, UnitPrice,
           ETLLoadID, LoadDate, UpdateDate
FROM       dbo.FactOnlineSales;
GO 11

-- Delete one percent of the rows, throughout all rowgroups
DELETE FROM dbo.FactOnlineSales2
WHERE       OnlineSalesKey % 100 = 33;

-- Update twenty percent of the rows, throughout all rowgroups
UPDATE     dbo.FactOnlineSales2
SET        UpdateDate = CURRENT_TIMESTAMP
WHERE      OnlineSalesKey % 5 = 2;
GO

-- Add 1,100,000 rows, in 11 batches of 100,000 each
INSERT     dbo.FactOnlineSales2
          (DateKey, StoreKey, ProductKey, PromotionKey,
           CurrencyKey, CustomerKey, SalesOrderNumber,
           SalesOrderLineNumber, SalesQuantity, SalesAmount,
           ReturnQuantity, ReturnAmount, DiscountQuantity,
           DiscountAmount, TotalCost, UnitCost, UnitPrice,
           ETLLoadID, LoadDate, UpdateDate)
SELECT TOP(100000)
           DateKey, StoreKey, ProductKey, PromotionKey,
           CurrencyKey, CustomerKey, SalesOrderNumber,
           SalesOrderLineNumber, SalesQuantity, SalesAmount,
           ReturnQuantity, ReturnAmount, DiscountQuantity,
           DiscountAmount, TotalCost, UnitCost, UnitPrice,
           ETLLoadID, LoadDate, UpdateDate
FROM       dbo.FactOnlineSales;
GO 11

-- Create a clustered index on the desired column
CREATE CLUSTERED INDEX CCI_FactOnlineSales2
ON dbo.FactOnlineSales2(ProductKey);

-- Finally, recreate the clustered columnstore index
CREATE CLUSTERED COLUMNSTORE INDEX CCI_FactOnlineSales2
ON dbo.FactOnlineSales2
WITH (DROP_EXISTING = ON, MAXDOP = 1);
GO
